<?php
namespace lc;

class Settings_Page {

  private $settings_api;
  private $config;
  private $section_id = 'options';
  private $license_status;

  function __construct( $args, $status ) {
    $this->config = $args;
    $this->settings_api = new \WeDevs_Settings_API();
    $this->settings_fields = array();
    $this->license_status = $status;

    // Create a settings page using wordpress-settings-api-class (Settings > Settings API)
    add_action( 'admin_init', array( $this, 'wpsac_admin_init' ) );
    add_action( 'admin_menu', array( $this, 'wpsac_admin_menu' ) );
    add_action('admin_init', array($this, 'restriction_plugin_settings'));
  }


  /**
   * Initialize wordpress-settings-api-class
   */
  public function wpsac_admin_init() {
    $this->settings_api->set_sections( $this->get_settings_sections() );
    $this->settings_api->set_fields( $this->get_settings_fields() );
    $this->settings_api->admin_init();
  }

  /**
   * Add menu link in WP Admin
   */
  public function wpsac_admin_menu() {
    add_menu_page(
        __( 'lc Order Restriction', 'lc_plugin' ),
        __( 'lc Plugin', 'lc_plugin' ),
        'manage_options',
        'lc_plugin-main-menu',
        array( $this, 'create_wpsac_main_menu_page' ),
        plugins_url( '../assets/icon.svg', __FILE__ ),
        20 // Adjust the position to avoid conflicts
    );

    add_submenu_page(
        'lc_plugin-main-menu',
        __( 'Setting', 'lc_plugin' ),
        __( 'Setting', 'lc_plugin' ),
        'manage_options',
        'lc_plugin-main-menu', // Use the same slug as the main menu
        array( $this, 'create_wpsac_main_menu_page' )
    );

    add_submenu_page(
        'lc_plugin-main-menu',
        __( 'Licensing', 'lc_plugin' ),
        __( 'Licensing', 'lc_plugin' ),
        'manage_options',
        'lc_plugin-licensing',
        array( $this, 'create_wpsac_licensing_page' )
    );
}


  /**
   * Callback for the main menu page
   */
  public function create_wpsac_main_menu_page() {
    ?> 
      <?php $this->settings_api->show_navigation(); ?>
      <div class="wrap">
        <?php settings_errors(); ?>
        <form method="post" action="options.php">
            <?php
            settings_fields('lc_plugin-main-menu');
            do_settings_sections('lc_plugin-main-menu');
            submit_button('Save Settings');
            ?>
        </form>
      </div>
    <?php
}


  /**
   * Retrieve configuration sections
   */
  public function get_settings_sections() {
    $sections = array(
      array(
        'id'    => $this->config['prefix'] . '_options',
        'title' => $this->config['short_name'] . ' ' . __( 'Licensing', 'lc_plugin' )
      )
    );

    return $sections;
  }

  /**
   * Returns all the settings fields
   *
   * @return array Settings fields
   */
  public function get_settings_fields() {
    $settings_fields = array(
        $this->config['prefix'] . '_options' => array(
            array(
                'name'              => $this->config['prefix'] . '_license_key',
                'label'             => __( 'License Key', 'lc-license-key' ),
                'desc'              => __( 'Enter the license key which you got after purchase.', 'lc-license-key' ),
                'type'              => 'text',
                'sanitize_callback' => array($this, 'sanitize_license_key'),
                'validate_callback' => array($this, 'validate_license_key'),
            )
        )
    );
    return $settings_fields;
  }

  public function sanitize_license_key($input) {
    // Sanitize and save the license key to the database
    return sanitize_text_field($input);
  }

  public function validate_license_key($input) {
    // Perform validation if needed
    return $input;
  }



  /**
   * Callback function for the settings submenu page
   */
  public function create_wpsac_licensing_page() {
    ?>
    <div class="wrap">
      <?php settings_errors(); ?>
      <form method="post" action="options.php">
        <?php
        $this->settings_api->show_navigation();
        $this->settings_api->show_forms();
        ?>
      </form>
    </div>
    <div class="license-status">
      <h4>License Status:- </h4><a href="https://dash.licensecontrol.net/index.php/store/wordpress-theme-plugin-lc/lc-plugin-protect-your-fake-order">Get License</a> 
      <?php
      if ($this->license_status == 'Active') {
        echo '<p style="color: white; background: #00a550; width:100px; padding:5px; border-radius: 15px; text-align: center;">Activated</p>';
      } else {
        echo '<p style="color: white; background: red; width: 100px; padding: 5px; border-radius: 15px; text-align: center;">' . esc_html($this->license_status) . '</p>';
      }
      ?>
    </div>
  <?php
  }


  private function prefix( $key = '', $after = '_' ) {
    return $this->config['prefix'] . $after . $key;
  }



  /// Settings functions
  // Run the actual plugin logic when the license is valid
  function restriction_plugin_settings() {
    add_settings_section('restriction_plugin_section', 'lc PLUGIN Setting', array($this, 'restriction_plugin_section_callback'), 'lc_plugin-main-menu');
    add_settings_field('enable_phone_restriction', 'Enable Mobile Number Restriction', array($this, 'enable_phone_restriction_callback'), 'lc_plugin-main-menu', 'restriction_plugin_section');
    register_setting('lc_plugin-main-menu', 'enable_phone_restriction');
    add_settings_field('enable_ip_restriction', 'Enable IP Address Restriction', array($this, 'enable_ip_restriction_callback'), 'lc_plugin-main-menu', 'restriction_plugin_section');
    register_setting('lc_plugin-main-menu', 'enable_ip_restriction');
    add_settings_field('order_time_period', 'Order Time Period (in days)', array($this, 'order_time_period_callback'), 'lc_plugin-main-menu', 'restriction_plugin_section');
    register_setting('lc_plugin-main-menu', 'order_time_period', array($this, 'sanitize_time_period'));
    add_settings_field('max_orders_within_period', 'Maximum Orders Allowed Within Time Period', array($this, 'max_orders_within_period_callback'), 'lc_plugin-main-menu', 'restriction_plugin_section');
    register_setting('lc_plugin-main-menu', 'max_orders_within_period', array($this, 'sanitize_max_orders'));

    add_settings_field('enable_vpn_restriction', 'Enable VPN/Proxy Restriction', array($this, 'enable_vpn_restriction_callback'), 'lc_plugin-main-menu', 'restriction_plugin_section');
    register_setting('lc_plugin-main-menu', 'enable_vpn_restriction');
    add_settings_field('vpn_api', 'VPN/Proxy API Key', array($this, 'vpn_api_callback'), 'lc_plugin-main-menu', 'restriction_plugin_section');
    register_setting('lc_plugin-main-menu', 'vpn_api', array($this, 'sanitize_vpn_api'));
  }
  
  
  function phone_restriction_section_callback() {
    echo '<p>Configure phone number restriction settings here.</p>';
  }

  function enable_phone_restriction_callback() {
    $phone_restriction_enabled = get_option('enable_phone_restriction', false);
    $checked = checked($phone_restriction_enabled, true, false);

    // Create a checkbox field for enabling or disabling phone number restriction
    echo '<input type="checkbox" id="enable_phone_restriction" name="enable_phone_restriction" value="1" ' . $checked . ' />';
    echo '<label for="enable_phone_restriction">Enable Mobile Number Restriction</label>';
  }


  function restriction_plugin_section_callback() {
    echo '<p>Order Restriction Setting here.</p>';
  }

  function max_orders_within_period_callback() {
    $max_orders = get_option('max_orders_within_period', 1); // Default to 1 order
    echo '<input type="number" id="max_orders_within_period" name="max_orders_within_period" value="' . $max_orders . '" min="1" />';
  }

  function vpn_api_callback() {
    $vpn_api = get_option('vpn_api', ""); // Default to 1 order
    echo '<input type="text" id="vpn_api" name="vpn_api" placeholder="Enter VPN/Proxy API key" value="' . $vpn_api . '" />';
    echo '<br/> <a target="_blank" href="https://licensecontrol.net/vpnapi">Get API</a>';
  }

  function sanitize_max_orders($input) {
    $input = absint($input); // Ensure it's a non-negative integer
    return $input;
  }

  function sanitize_vpn_api($input) {
    // $input = absint($input); // made this function for future plan
    return $input;
  }

  function order_time_period_callback() {
    $time_period = get_option('order_time_period', 0);
    echo '<select id="order_time_period" name="order_time_period">';
    $options = array(
        0 => 'Disable',
        1 => '1 Day',
        2 => '2 Days',
        3 => '3 Days',
        4 => '4 Days',
        5 => '5 Days',
        6 => '6 Days',
        7 => '7 Days',
        8 => '8 Days',
        9 => '9 Days',
        10 => '10 Days',
        30 => '1 Month',
        90 => '3 Months',
        180 => '6 Months',
        365 => '1 Year'
    );
    foreach ($options as $key => $label) {
        echo '<option value="' . $key . '" ' . selected($time_period, $key, false) . '>' . $label . '</option>';
    }
    echo '</select>';
}


  function enable_ip_restriction_callback() {
    $ip_restriction_enabled = get_option('enable_ip_restriction', false);
    $checked = checked($ip_restriction_enabled, true, false);
    echo '<input type="checkbox" id="enable_ip_restriction" name="enable_ip_restriction" value="1" ' . $checked . ' />';
    echo '<label for="enable_ip_restriction">Enable IP Address Restriction</label>';
  }

  function enable_vpn_restriction_callback() {
    $phone_restriction_enabled = get_option('enable_vpn_restriction', false);
    $checked = checked($phone_restriction_enabled, true, false);

    // Create a checkbox field for enabling or disabling vpn restriction
    echo '<input type="checkbox" id="enable_vpn_restriction" name="enable_vpn_restriction" value="1" ' . $checked . ' />';
    echo '<label for="enable_vpn_restriction">Enable VPN/Proxy Restriction</label>';
  }

  // Sanitize the time period option
  function sanitize_time_period($input) {
    $input = absint($input); // Ensure it's a non-negative integer
    return $input;
  }

}
